﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/networkmanager/NetworkManagerRequest.h>
#include <aws/networkmanager/NetworkManager_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace NetworkManager {
namespace Model {

/**
 */
class GetLinksRequest : public NetworkManagerRequest {
 public:
  AWS_NETWORKMANAGER_API GetLinksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetLinks"; }

  AWS_NETWORKMANAGER_API Aws::String SerializePayload() const override;

  AWS_NETWORKMANAGER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the global network.</p>
   */
  inline const Aws::String& GetGlobalNetworkId() const { return m_globalNetworkId; }
  inline bool GlobalNetworkIdHasBeenSet() const { return m_globalNetworkIdHasBeenSet; }
  template <typename GlobalNetworkIdT = Aws::String>
  void SetGlobalNetworkId(GlobalNetworkIdT&& value) {
    m_globalNetworkIdHasBeenSet = true;
    m_globalNetworkId = std::forward<GlobalNetworkIdT>(value);
  }
  template <typename GlobalNetworkIdT = Aws::String>
  GetLinksRequest& WithGlobalNetworkId(GlobalNetworkIdT&& value) {
    SetGlobalNetworkId(std::forward<GlobalNetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more link IDs. The maximum is 10.</p>
   */
  inline const Aws::Vector<Aws::String>& GetLinkIds() const { return m_linkIds; }
  inline bool LinkIdsHasBeenSet() const { return m_linkIdsHasBeenSet; }
  template <typename LinkIdsT = Aws::Vector<Aws::String>>
  void SetLinkIds(LinkIdsT&& value) {
    m_linkIdsHasBeenSet = true;
    m_linkIds = std::forward<LinkIdsT>(value);
  }
  template <typename LinkIdsT = Aws::Vector<Aws::String>>
  GetLinksRequest& WithLinkIds(LinkIdsT&& value) {
    SetLinkIds(std::forward<LinkIdsT>(value));
    return *this;
  }
  template <typename LinkIdsT = Aws::String>
  GetLinksRequest& AddLinkIds(LinkIdsT&& value) {
    m_linkIdsHasBeenSet = true;
    m_linkIds.emplace_back(std::forward<LinkIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the site.</p>
   */
  inline const Aws::String& GetSiteId() const { return m_siteId; }
  inline bool SiteIdHasBeenSet() const { return m_siteIdHasBeenSet; }
  template <typename SiteIdT = Aws::String>
  void SetSiteId(SiteIdT&& value) {
    m_siteIdHasBeenSet = true;
    m_siteId = std::forward<SiteIdT>(value);
  }
  template <typename SiteIdT = Aws::String>
  GetLinksRequest& WithSiteId(SiteIdT&& value) {
    SetSiteId(std::forward<SiteIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The link type.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  GetLinksRequest& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The link provider.</p>
   */
  inline const Aws::String& GetProvider() const { return m_provider; }
  inline bool ProviderHasBeenSet() const { return m_providerHasBeenSet; }
  template <typename ProviderT = Aws::String>
  void SetProvider(ProviderT&& value) {
    m_providerHasBeenSet = true;
    m_provider = std::forward<ProviderT>(value);
  }
  template <typename ProviderT = Aws::String>
  GetLinksRequest& WithProvider(ProviderT&& value) {
    SetProvider(std::forward<ProviderT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetLinksRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetLinksRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_globalNetworkId;

  Aws::Vector<Aws::String> m_linkIds;

  Aws::String m_siteId;

  Aws::String m_type;

  Aws::String m_provider;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_globalNetworkIdHasBeenSet = false;
  bool m_linkIdsHasBeenSet = false;
  bool m_siteIdHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_providerHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkManager
}  // namespace Aws
